import java.util.*;
import java.io.*;

abstract class InfiniteExemplarModel {

    /*****************************************************************/
    /*                     Instance Variables                        */
    /*****************************************************************/

    // similarities_(i,j) is the affinity of point i to exemplar j 
    // (not necessarily symmetric)
    protected double similarities_[][];

    protected int numVars_;

    protected int[] trueLabels_;

    // Dirichlet Process concentration parameter
    public double alpha_;

    // For logging
    protected PrintStream log_;

    /*****************************************************************/
    /*                       Abstract Methods                        */
    /*****************************************************************/    
    
    /**
     * Actually do the inference.  The goal is to come up with a single,
     * best clustering.
     */
    abstract void MAPInference();

    /**
     * Choose which points are exemplars.
     */
    abstract ArrayList<Integer> CurrentExemplars();

    /**
     * Find an assignment for all points.
     */
    abstract int[] CurrentAssignments();

    /**
     * Print out details of inference.
     */
    abstract void PrintStats();

    /**
     * Take a parameter object and set internal parameters accordingly.
     */
    abstract void LoadParametersFromFile(String filename);

    /**
     * Keep track of the name of the algorithm being implemented.
     */
    abstract String FullName();
    abstract String ShortName();
    

    /*****************************************************************/
    /*                      Implemented Methods                      */
    /*****************************************************************/    

    /**
     * Slightly more flexible logging functions.
     */
    public void LogToSystemOut() {
	log_ = System.out;
    }
    public void LogToSystemErr() {
	log_ = System.err;
    }
    public void LogToFile(String filename) {
	
    }

    public void LogLine(String text) {
	log_.println(text);
    }
    public void LogText(String text) {
	log_.print(text);
    }

    /**
     * Return the true assignment.
     */
    public int[] TrueLabels() {
	return trueLabels_;
    }

    public boolean MessagesConverged() {
	return false;
    }

    /**
     * Display an assignment along with its log probability.
     */
    public void PrintAssignment(int[] a) {
	double fullLogProb = LogProb(a);
	System.err.print(fullLogProb + " [ ");
	for (int i = 0; i < numVars_; i++) {
	    System.err.print(a[i] + " ");
	}
	System.err.println("]");
    }

    /**
     * Get number of points per cluster.
     */
    public int[] ClusterSizes(int[] assignment) {
	int[] nIs = new int[numVars_];
	for (int i = 0; i < numVars_; i++) nIs[i] = 0;

	for (int i = 0; i < numVars_; i++) {
	    int cI = assignment[i];
	    if (cI != -1) {
		nIs[cI]++;
	    }
	}
	return nIs;
    }

    /**
     * Compute the log probability of a given assignment under this
     * model.
     */
    public double LogProb(int[] assignment) {
	// First take similarities terms
	assert(assignment.length == numVars_);
	double logProb = 0;

	// Create and zero out counts of cluster sizes
	int[] nIs = ClusterSizes(assignment);

	for (int i = 0; i < numVars_; i++) {
	    if (assignment[i] == -1) {
		logProb -= Double.MAX_VALUE;
	    } else {
		logProb += similarities_[i][assignment[i]];
	    }
	}

	for (int i = 0; i < numVars_; i++) {
	    if (nIs[i] > 0) {
		logProb += DPUtils.LogGamma(nIs[i]) - Math.log(nIs[i]);
	    }
	}

	// check consistency
	for (int i = 0; i < numVars_; i++) {
	    if (assignment[i] == -1 || 
		assignment[assignment[i]] != assignment[i]) {
		logProb = -Double.MAX_VALUE;
	    }
	}

	return logProb;
    }

    /**
     * Compute the rand index between two assignments.
     */
    static public double RandIndex(int[] a1, int[] a2) {
	assert(a1.length == a2.length);

	// Look at all pairs and see if they belong in the same group
	int numerator = 0;
	int denominator = 0;
	for (int i = 0; i < a1.length; i++) {
	    for (int j = i+1; j < a1.length; j++) {
		if ((a1[i] == a1[j] && a2[i] == a2[j]) ||
		    (a1[i] != a1[j] && a2[i] != a2[j])) {
		    numerator++;
		}
		denominator++;
	    }
	}
	return numerator / (double) denominator;
    }

    /**
     * Load the true labels from a file.
     * WARNING: Assumes file was generated by MATLAB and is thus 1-indexed.
     */
    public void LoadTrueLabelsFromFile(String filename) {
	System.err.println("Loading true labels from " + filename);

	// Allocate space
	trueLabels_ = new int[numVars_];

	try {
	    BufferedReader input = 
		new BufferedReader(new FileReader(filename));

	    // There's just one line to read
	    String line = input.readLine();
	    assert(line != null);
	    
	    String[] entries = line.trim().split("\\s+");
	    for (int j = 0; j < numVars_; j++) {
		trueLabels_[j] = (int) Double.parseDouble(entries[j]) - 1;
	    }
	}
	catch (Exception ex){
	    System.err.println("Failed loading true labels");
	    for (int i = 0; i < trueLabels_.length; i++) {
		trueLabels_[i] = 1;
	    }
	    //ex.printStackTrace();
	    //System.exit(1);
	}	

    }

    /**
     * Load similarities from an existing infinite exemplar model.
     */
    public void LoadSimilaritiesFromSimilarities(double[][] s) {
	for (int i = 0; i < numVars_; i++) {
	    for (int j = 0; j < numVars_; j++) {
		similarities_[i][j] = s[i][j];
	    }
	}
    }

    /**
     * Add a constant to the diagonal of the similarities.
     */
    public void AddToDiagonal(double delta) {
	for (int i = 0; i < numVars_; i++) {
	    similarities_[i][i] += delta;
	}
    }

    /**
     * Add a constant to the diagonal of the similarities.
     */
    public void MultiplySimilarities(double s) {
	for (int i = 0; i < numVars_; i++) {
	    for (int j = 0; j < numVars_; j++) {
		similarities_[i][j] *= s;
	    }
	}
    }

    /**
     * Load data from a file where there are N rows of D dimensional points.
     */
    public void LoadSimilaritiesFromFile(String filename) {
	try {
	    BufferedReader input = new BufferedReader(new FileReader(filename));

	    System.err.println("Loading " + numVars_ + "x" + numVars_
	    		       + " similarity matrix " + " from " + filename);

	    String line;
	    for (int i = 0; i < numVars_; i++) {
		line = input.readLine();
		assert(line != null);

		String[] entries = line.trim().split("\\s+");
		for (int j = 0; j < numVars_; j++) {
		    similarities_[i][j] = Double.parseDouble(entries[j]);
		}
	    }
	}
	catch (Exception ex){
	    ex.printStackTrace();
	    System.exit(1);
	}
    }

    /**
     * Make up some simple test weights, just to get the code running.
     */
    public void LoadTestSimilarities() {
	System.out.println("Loading test similarities...");

	// Randomize based on date/time
        Date date = new Date();
        long time = date.getTime();
	Random r = new Random(time);

	// Generate random weights
	for (int i = 0; i < numVars_; i++) {
	    for (int j = 0; j < numVars_; j++) {
		if (i < numVars_ / 2 && j < numVars_ / 2) {
		    similarities_[i][j] = 1;
		} else if (i >= numVars_ / 2 && j >= numVars_ / 2) {
		    similarities_[i][j] = 1;
		} else {
		    similarities_[i][j] = 0;
		}

		if (i == j && i == 1) {
		    similarities_[i][j] += 2;
		} else if (i == j && i == numVars_ - 1) {
		    similarities_[i][j] += 1;
		}
		similarities_[i][j] += Double.MIN_VALUE * 
		    100 * r.nextGaussian();

		System.out.print(similarities_[i][j] + " ");
	    }
	    System.out.println();
	}
    }

}
